/************************************************************************
 *
 * \file: Server.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Demo application
 *
 * \author: J. Harder / ADITG/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <adit_logging.h>
#include <aauto_macros.h>
#include <aauto/AautoLogging.h>
#include "Server.h"
#include "DemoFactory.h"
#include "utils/PfCfgConfiguration.h"
#include "Testbed.h"
#include "AutoSmoketest.h"

LOG_IMPORT_CONTEXT(demo)

#define ACCESSORY_MANUFACTURER_NAME     "Android"
#define ACCESSORY_MODEL_NAME            "Android Auto"
#define ACCESSORY_DESCRIPTION           "Android Auto"
#define ACCESSORY_VERSION               "1.3"
#define ACCESSORY_URI                   "http://www.android.com/auto"
#define ACCESSORY_SERIAL_NUMBER         "000000001234567"

namespace adit { namespace aauto {

using namespace utility;

Server::Server()
{
    static int _counter = 0;
    _counter++;
    if (_counter > 1)
        LOG_ERROR((demo, "Server object shall not be used twice"));

    running = false;
    mConfig = nullptr;

    DemoFactory::instance().setLogger(
            [](const char* msg) { LOGD_DEBUG((demo, "%s", msg)); },
            [](const char* msg) { LOG_ERROR((demo, "%s", msg)); });
            
    pthread_mutex_init(&sessionMutex, NULL);
}

Server::~Server()
{
    stopEventDispatcher();
    pthread_mutex_destroy(&sessionMutex);
}

// the function is used to call the entry point function pointer with arguments
static bool EntryPointCallback(void* inEntryPoint)
{
    auto fn = reinterpret_cast<void (*)(FactoryRegisterFn)>(inEntryPoint);
    fn(DemoFactory::staticRegister);
    return true;
}

static void ExitCallback(void* inExitPoint)
{
    auto fn = reinterpret_cast<void (*)(void)>(inExitPoint);
    fn();
}

bool Server::init(const std::string& inConfigPath)
{
    // register all AAUTO DLT context of aauto/platform
    // TODO different function name?
    aautoRegisterCtxtWithDLT();

    LOG_INFO((demo, "aauto-demo git tag: %s", COMP_GIT_VERSION));
    LOG_INFO((demo, "Google protobuf version: 0x%X | %d", GOOGLE_PROTOBUF_VERSION,
            GOOGLE_PROTOBUF_VERSION));

    if (!startEventDispatcher())
    {
        LOG_WARN((demo, "StartEvent Dispatcher Failed"));
    }
    std::string cfgFile = "";
    if (!inConfigPath.empty()) {
        cfgFile = inConfigPath;
    } else {
        cfgFile = "/etc/pfcfg/aauto-demo.cfg";
        LOG_WARN((demo, "Path to config file not set. use default path %s", cfgFile.c_str()));
    }
    /* read configuration file */
    mConfig = PfCfgConfiguration::FromFile(cfgFile.c_str());
    if (mConfig != nullptr) {
        LOG_INFO((demo, "found and open config file %s", cfgFile.c_str()));
    } else {
        LOG_ERROR((demo, "could not open config file %s", cfgFile.c_str()));
        return false;
    }

    /* read libraries which should be loaded from configuration */
    auto libraries = mConfig->GetItems("libraries");
    /* load libraries */
    for (auto library : libraries)
    {
        if (!pluginManager.load(library, "aauto_LibraryEntryPoint",
                                EntryPointCallback, "aauto_LibraryExitPoint", ExitCallback))
        {
            // error occurred and logged
            // continue anyway
        }
    }

    if(Testbed::instance().getTestbedMode())
    {
        // Fix layer id and surface id
        mConfig->SetItem("aauto-demo-layer-id", std::to_string(Testbed::instance().getLayerID()));
        mConfig->SetItem("aauto-demo-surface-id", std::to_string(Testbed::instance().getSurfaceID()));
    }

    /* create aauto-demo layer manager */
    layerManager = new aautoLayerManager(*mConfig);
    if (layerManager == nullptr)
    {
        LOG_ERROR((demo, "could not allocate resource of aautoLayerManager"));
    }

    running = true;
    return true;
}

void Server::cleanUp()
{
    running = false;
    aauto_safe_delete(mConfig);

    pluginManager.unloadAll();

    if (layerManager != nullptr)
    {
        LOGD_DEBUG((demo, "finalize layer setting"));
        layerManager->finalize();
    }

    // TODO different function name?
    aautoDeRegisterCtxtWithDLT();
}

void Server::requestStop()
{
    for(std::map<std::string, ::shared_ptr<Session>>::iterator iter = mSessions.begin(); iter != mSessions.end(); ++iter)
    {
        //don't send ByeBye when in automatic test mode
        if(AutoSmoketest::instance().getTestMode()!=AUTOMATIC)
        {
            LOGD_DEBUG((demo, "Server::requestStop()  send ByeByeRequest"));
            iter->second->sendByeByeRequest();
        }
    }
    LOGD_DEBUG((demo, "Server::requestStop()  destroy all active session"));
    destroySession();
    mSessions.clear();
}

bool Server::switchDevice(uint32_t inVid, uint32_t inPid, std::string inSerial,
                          aoapTransportInfo_t* outAoapTransportInfo)
{
    aauto_return_value_on_null(aauto_demo, outAoapTransportInfo, false);
    bool ret = false;
    const unsigned int timeout = 10000;

    // fill AccessoryInfo to identify & start the AndroidAuto application on the MD
    aoapDeviceInfo_t aoapDeviceInfo;
    aoapDeviceInfo.aoapAccessoryInfo.manufacturer = ACCESSORY_MANUFACTURER_NAME;
    aoapDeviceInfo.aoapAccessoryInfo.modelName = ACCESSORY_MODEL_NAME;
    aoapDeviceInfo.aoapAccessoryInfo.description = ACCESSORY_DESCRIPTION;
    aoapDeviceInfo.aoapAccessoryInfo.version = ACCESSORY_VERSION;
    aoapDeviceInfo.aoapAccessoryInfo.uri = ACCESSORY_URI;
    aoapDeviceInfo.aoapAccessoryInfo.serial = ACCESSORY_SERIAL_NUMBER;
    /* Enable Audio support.
     * AOA 2.0 includes optional the support for audio output from the MD to the HU.
     * AOA 1.0 does not supports the optional audio output.
     * Because AndroidAuto supports AOA 1.0, set enableAudio to zero. */
    aoapDeviceInfo.aoapAccessoryInfo.enableAudio = 0;

    // fill aoapDeviceInfo_t structure to switch the MD into AndroidAuto mode
    aoapDeviceInfo.vendorId = inVid;
    aoapDeviceInfo.productId = inPid;
    // TODO who does free this?
    aoapDeviceInfo.pSerial = strdup(inSerial.c_str());

    AoapDevice aoapDevice;
    /* switch the MD into AndroidAuto mode.
     * this will not start the communication with between the MD and the GalReceiver. */
    int result = aoapDevice.switchDevice(&aoapDeviceInfo, outAoapTransportInfo, timeout);

    if (0 == result) {
        LOG_INFO((demo, "Start AndroidAuto on device (vendorId=%X, producId=%X, serial=%s)",
                aoapDeviceInfo.vendorId, aoapDeviceInfo.productId, aoapDeviceInfo.pSerial));
        ret = true;
    }
    else {
        LOG_ERROR((demo, "Failed to start AndroidAuto on device (vendorId=%X, producId=%X, " \
                "serial=%s) = %d",
                aoapDeviceInfo.vendorId, aoapDeviceInfo.productId, aoapDeviceInfo.pSerial,
                result));
        ret = false;
    }

    free(aoapDeviceInfo.pSerial);
    return ret;
}

::shared_ptr<Session> Server::createSession(const std::string& inSerial, aoapTransportInfo_t inAoapTransportInfo, TestParameter inParameter)
{
    auto found = mSessions.find(inSerial);
    if (found != mSessions.end()) {
        LOG_ERROR((demo, "Session for serial=%s already created!", inSerial.c_str()));
        mSessions.erase(found);
    }

    if (!layerManager->addMobileDevice(inSerial))
    {
        LOG_ERROR((demo, "Could not add new mobile device"));
    }

    mSessions.insert(std::pair<std::string, ::shared_ptr<Session>>(inSerial, new Session(*mConfig, inSerial)));
    int ret = 0;
    /* start session and initialize the GalReceiver */
    found = mSessions.find(inSerial);
    if (found != mSessions.end()) {
        found->second->setTestParameter(inParameter);
        found->second->setLayer(layerManager, inSerial);
        if(!found->second->Start()) {
            // TODO error handling
            LOG_ERROR((demo, "Start Sever and/or initialize GalReceiver failed"));
            ret = -1;
        } else {
            /* create abstract Transport and override with AOAP specific implementation
             * to establish data transfer between MD and HU. */
            ::shared_ptr<Transport> transport =
                    new AoapTransport(found->second->GetReceiver(), &inAoapTransportInfo);
            /* Start the GalReceiver transport */
            if (transport->start())
            {
                LOG_INFO((demo, "Start GalReceiver transport successes"));

                // attach transport to the session and don't care about it anymore
                if(found->second->attachTransport(transport))
                {
                    ret = 0;
                }
                else
                {
                    LOG_ERROR((demo, "attachTransport failed"));
                    ret = -1;
                }
            }
            else
            {
                LOG_ERROR((demo, "Start GalReceiver transport failed"));
                ret = -1;
            }
        }
    } else {
        LOG_ERROR((demo, "Session created and add, but not found Oo !!"));
        return nullptr;
    }

    if (ret != 0)
        return nullptr;

    // add to internal session list
    return found->second;
}

::shared_ptr<Session> Server::getSession(const std::string& inSerial)
{
    auto found = mSessions.find(inSerial);
    if (found != mSessions.end()) {
        return found->second;
    } else {
        LOG_WARN((demo, "Session for serial=%s not found!", inSerial.c_str()));
    }
    return nullptr;
}

void Server::destroySession(void)
{
    for(std::map<std::string, ::shared_ptr<Session>>::iterator iter = mSessions.begin(); iter != mSessions.end(); )
    {
        LOG_INFO((demo, "release session for serial=%s", iter->first.c_str()));
        iter->second->requestStop();
        iter->second->waitForExit();
        /* delete MD from AAutoLayerManager */
        layerManager->deleteMobileDevice(iter->first);
        /* erase AAUTO session from map */
        iter = mSessions.erase(iter);
    }
}

void Server::destroySession(const std::string& inSerial)
{
    auto found = mSessions.find(inSerial);
    if (found != mSessions.end()) {
        found->second->requestStop();
        found->second->waitForExit();
        mSessions.erase(found);
    } else {
        LOG_WARN((demo, "Session for serial=%s not found!", inSerial.c_str()));
    }

    layerManager->deleteMobileDevice(inSerial);
}

void Server::onByeByeRequest(std::string inSerial)
{
    LOG_INFO((demo, "Server::%s()  Destroy AndroidAuto session for device with serial %s", __func__, inSerial.c_str()));
    destroySession(inSerial);
}

} } // namespace adit { namespace aauto {
